import pandas as pd
import tkinter as tk
from tkinter import ttk, filedialog, messagebox

class PDCApp:
    def __init__(self, root):
        self.root = root
        self.root.title("Calcolo Medie PDC per Code")
        self.root.geometry("950x650")
        self.df = None
        self.risultati_df = None

        # Titolo
        tk.Label(root, text="📊 Calcolo Medie e Medie Ponderate per CODE",
                 font=("Helvetica", 16, "bold")).pack(pady=10)

        # Pulsante per caricare file
        tk.Button(root, text="📂 Carica File Excel", command=self.carica_file,
                  bg="#4CAF50", fg="white", width=25).pack(pady=10)

        # Etichetta file
        self.file_label = tk.Label(root, text="Nessun file caricato", fg="gray")
        self.file_label.pack(pady=5)

        # Tabella risultati
        self.tree = ttk.Treeview(
            root,
            columns=("code", "media_pdc", "media_pdc_corr", "media_pond_pdc", "media_pond_pdc_corr"),
            show="headings",
            height=15
        )
        self.tree.pack(padx=10, pady=10, fill="both", expand=True)

        for col, text in zip(
            ("code", "media_pdc", "media_pdc_corr", "media_pond_pdc", "media_pond_pdc_corr"),
            ("CODE", "Media PDC", "Media PDC corretta", "Media ponderata PDC", "Media ponderata PDC corretta")
        ):
            self.tree.heading(col, text=text)
            self.tree.column(col, anchor="center", width=180)

        # Pulsante per esportare risultati
        tk.Button(root, text="💾 Esporta Risultati in Excel", command=self.esporta_excel,
                  bg="#2196F3", fg="white", width=30).pack(pady=10)

        # Etichetta risultati globali
        self.global_label = tk.Label(root, text="", font=("Helvetica", 12, "bold"), justify="center")
        self.global_label.pack(pady=10)

    def carica_file(self):
        file_path = filedialog.askopenfilename(
            title="Seleziona un file Excel",
            filetypes=[("File Excel", "*.xlsx *.xls")]
        )
        if not file_path:
            return

        try:
            self.df = pd.read_excel(file_path)
            self.df.columns = [c.strip().lower() for c in self.df.columns]

            richieste = ["code", "pdc", "pdc corretta", "persistence"]
            mancanti = [col for col in richieste if col not in self.df.columns]
            if mancanti:
                messagebox.showerror("Errore", f"Mancano le colonne: {', '.join(mancanti)}")
                return

            self.file_label.config(text=f"File caricato: {file_path.split('/')[-1]}", fg="black")
            self.calcola_medie()
        except Exception as e:
            messagebox.showerror("Errore", f"Errore durante il caricamento del file:\n{e}")

    def calcola_medie(self):
        df = self.df.copy()

        # Filtra righe valide (code pieno)
        df = df[df["code"].notna() & (df["code"].astype(str).str.strip() != "")]
        if df.empty:
            messagebox.showwarning("Attenzione", "Nessuna riga con campo 'code' pieno.")
            return

        # Converti le colonne numeriche
        for col in ["pdc", "pdc corretta", "persistence"]:
            df[col] = pd.to_numeric(df[col], errors="coerce")

        df = df.dropna(subset=["pdc", "pdc corretta", "persistence"])
        if df.empty:
            messagebox.showwarning("Attenzione", "Dati non validi dopo la pulizia numerica.")
            return

        # Raggruppa per code
        gruppi = df.groupby("code").apply(self._calcola_statistiche).reset_index()

        # 🔹 Calcoli globali come medie delle medie per code
        media_glob_pdc = gruppi["media_pdc"].mean()
        media_glob_pdc_corr = gruppi["media_pdc_corr"].mean()
        media_pond_glob_pdc = gruppi["media_pond_pdc"].mean()
        media_pond_glob_pdc_corr = gruppi["media_pond_pdc_corr"].mean()

        self.risultati_df = gruppi  # Salva per eventuale esportazione

        # Mostra tabella
        for row in self.tree.get_children():
            self.tree.delete(row)
        for _, r in gruppi.iterrows():
            self.tree.insert("", "end", values=(
                r["code"],
                f"{r['media_pdc']:.4f}",
                f"{r['media_pdc_corr']:.4f}",
                f"{r['media_pond_pdc']:.4f}",
                f"{r['media_pond_pdc_corr']:.4f}"
            ))

        # Mostra medie globali (basate sulle medie per code)
        self.global_label.config(text=(
            f"🌐 MEDIE GLOBALI (medie delle medie per CODE)\n"
            f"Media PDC: {media_glob_pdc:.4f} | "
            f"Media PDC corretta: {media_glob_pdc_corr:.4f}\n"
            f"Media ponderata PDC: {media_pond_glob_pdc:.4f} | "
            f"Media ponderata PDC corretta: {media_pond_glob_pdc_corr:.4f}"
        ))

    def _calcola_statistiche(self, group):
        """Calcola media e media ponderata per un singolo gruppo (code)."""
        media_pdc = group["pdc"].mean()
        media_pdc_corr = group["pdc corretta"].mean()

        if group["persistence"].sum() == 0:
            media_pond_pdc = float('nan')
            media_pond_pdc_corr = float('nan')
        else:
            media_pond_pdc = (group["pdc"] * group["persistence"]).sum() / group["persistence"].sum()
            media_pond_pdc_corr = (group["pdc corretta"] * group["persistence"]).sum() / group["persistence"].sum()

        return pd.Series({
            "media_pdc": media_pdc,
            "media_pdc_corr": media_pdc_corr,
            "media_pond_pdc": media_pond_pdc,
            "media_pond_pdc_corr": media_pond_pdc_corr
        })

    def esporta_excel(self):
        if self.risultati_df is None or self.risultati_df.empty:
            messagebox.showwarning("Attenzione", "Nessun risultato da esportare.")
            return

        file_path = filedialog.asksaveasfilename(
            title="Salva risultati come",
            defaultextension=".xlsx",
            filetypes=[("File Excel", "*.xlsx")]
        )
        if not file_path:
            return

        try:
            # Salva risultati per code + globali
            with pd.ExcelWriter(file_path) as writer:
                self.risultati_df.to_excel(writer, index=False, sheet_name="Per_CODE")

                globali = pd.DataFrame([{
                    "Media PDC": self.risultati_df["media_pdc"].mean(),
                    "Media PDC corretta": self.risultati_df["media_pdc_corr"].mean(),
                    "Media ponderata PDC": self.risultati_df["media_pond_pdc"].mean(),
                    "Media ponderata PDC corretta": self.risultati_df["media_pond_pdc_corr"].mean()
                }])
                globali.to_excel(writer, index=False, sheet_name="Globali")

            messagebox.showinfo("✅ Esportazione completata", f"Risultati salvati in:\n{file_path}")
        except Exception as e:
            messagebox.showerror("Errore", f"Errore durante il salvataggio del file:\n{e}")

if __name__ == "__main__":
    root = tk.Tk()
    app = PDCApp(root)
    root.mainloop()
